import cv2
import numpy as np
from scipy.special import erf
import config_reader
from typing import Tuple, Optional


class ProjectionSplit:
    """
    Handles splitting images for dual-projector setups with overlap blending.
    """

    def __init__(self):
        self.image_left: Optional[np.ndarray] = None
        self.image_right: Optional[np.ndarray] = None
        self.image_main: Optional[np.ndarray] = None
        self.cfg = config_reader.ConfigReader("config.ini")
        print("ProjectionSplit initialized (NumPy backend)")

    def _generate_alpha_curves(
        self, overlap: int, blend_type: str
    ) -> Tuple[np.ndarray, np.ndarray]:
        x = np.linspace(0, 1, overlap)
        #gamma = self.cfg.get_gamma()   # e.g. 0.4 ~ 1.4
        gamma = 1
    
        x = x ** gamma

        if blend_type == "linear":
            param = self.cfg.get_linear_parameter()
            left = 1 - param * x
            right = 1 - param + param * x

        elif blend_type == "quadratic":
            left = (1 - x) ** 2
            right = x ** 2

        elif blend_type == "gaussian":
            sigma = 0.25
            g = 0.5 * (1 + erf((x - 0.5) / (sigma * np.sqrt(2))))
            left = 1 - g
            right = g
        else:
            raise ValueError(f"Unsupported blend_type: {blend_type}")

        return left.astype(np.float32), right.astype(np.float32)

    def _apply_blending(self, image: np.ndarray, overlap: int, blend_type: str) -> None:
        if image.shape[2] == 3:
            image = cv2.cvtColor(image, cv2.COLOR_BGR2BGRA)

        height, width = image.shape[:2]
        split_x = width // 2

        left_end = split_x + overlap // 2
        right_start = split_x - overlap // 2

        left_img = image[:, :left_end].astype(np.float32)
        right_img = image[:, right_start:].astype(np.float32)

        alpha_l, alpha_r = self._generate_alpha_curves(overlap, blend_type)

        # Apply alpha blending (NumPy)
        left_img[:, -overlap:, 3] *= alpha_l[None, :]
        right_img[:, :overlap, 3] *= alpha_r[None, :]

        self.image_left = left_img.astype(np.uint8)
        self.image_right = right_img.astype(np.uint8)

        cv2.imwrite("left.png", self.image_left)
        cv2.imwrite("right.png", self.image_right)

    def process_images(self, overlap: int = 75, blend_type: str = "linear"):
        image = cv2.imread("input.png", cv2.IMREAD_UNCHANGED)
        if image is None:
            raise FileNotFoundError("input.png not found.")
        self.image_main = image
        self._apply_blending(image, overlap, blend_type)
