import cv2
import numpy as np
import time
from typing import Tuple, Optional

class CalibrationManager:
    """
    Handles auto-calibration of projector overlap using a camera.
    """
    def __init__(self, camera_index: int = 0):
        self.camera_index = camera_index
        self.sift = cv2.SIFT_create()
        # FLANN parameters for matcher
        FLANN_INDEX_KDTREE = 1
        index_params = dict(algorithm=FLANN_INDEX_KDTREE, trees=5)
        search_params = dict(checks=50)
        self.flann = cv2.FlannBasedMatcher(index_params, search_params)

    def capture_image(self) -> Optional[np.ndarray]:
        """
        Captures a single frame from the camera.
        Returns None if capture fails.
        """
        cap = cv2.VideoCapture(self.camera_index)
        if not cap.isOpened():
            print(f"Error: Could not open camera {self.camera_index}")
            return None
        
        # Warmup
        for _ in range(10):
            cap.read()
            
        ret, frame = cap.read()
        cap.release()
        
        if not ret:
            print("Error: Could not read frame")
            return None
            
        return frame

    def calculate_overlap(self, img_left: np.ndarray, img_right: np.ndarray) -> Optional[int]:
        """
        Calculates the horizontal overlap between two images using SIFT features.
        img_left: The captured image when ONLY the left projector is on.
        img_right: The captured image when ONLY the right projector is on.
        """
        # Convert to grayscale
        gray_left = cv2.cvtColor(img_left, cv2.COLOR_BGR2GRAY)
        gray_right = cv2.cvtColor(img_right, cv2.COLOR_BGR2GRAY)

        # Detect keypoints and descriptors
        kp1, des1 = self.sift.detectAndCompute(gray_left, None)
        kp2, des2 = self.sift.detectAndCompute(gray_right, None)

        if des1 is None or des2 is None or len(kp1) < 2 or len(kp2) < 2:
            print("Not enough features detected.")
            return None

        # Match descriptors
        matches = self.flann.knnMatch(des1, des2, k=2)

        # Lowe's ratio test
        good_matches = []
        for m, n in matches:
            if m.distance < 0.7 * n.distance:
                good_matches.append(m)

        if len(good_matches) < 4:
            print("Not enough good matches found.")
            return None

        # Extract location of good matches
        pts_left = np.float32([kp1[m.queryIdx].pt for m in good_matches]).reshape(-1, 1, 2)
        pts_right = np.float32([kp2[m.trainIdx].pt for m in good_matches]).reshape(-1, 1, 2)

        # Find Homography
        H, mask = cv2.findHomography(pts_left, pts_right, cv2.RANSAC, 5.0)
        
        if H is None:
            print("Could not find homography.")
            return None

        # We are interested in the translation component, specifically horizontal shift.
        # H maps points from left image to right image.
        # If the right image is shifted to the right relative to the left image in the camera view...
        # Actually, let's think about the physical setup.
        # We have a screen.
        # Left projector projects on the left side.
        # Right projector projects on the right side.
        # There is an overlap region in the middle.
        # In the camera view, the "Left Projector Image" will appear on the left.
        # The "Right Projector Image" will appear on the right.
        # They share common features in the overlap area.
        # We want to find how many pixels *in the source image space* this corresponds to.
        # This is tricky because the camera pixels != projector pixels.
        
        # SIMPLIFICATION for this iteration:
        # We assume the camera sees the whole screen.
        # We need to map Camera Pixels -> Projector Pixels.
        # This usually requires a structured light calibration (Gray codes).
        # However, the user asked for "feature detection (SIFT/ORB)".
        # If we use SIFT on the *projected content*, we are matching the content itself.
        # If we project the SAME content (the original image) split into two, 
        # the features in the overlap area should be identical.
        
        # Let's refine the strategy:
        # 1. We have the original source image.
        # 2. We project Left half. Capture C_L.
        # 3. We project Right half. Capture C_R.
        # 4. We find matches between C_L and C_R.
        # 5. The matches should be in the overlap region.
        # 6. The distance between matched points in C_L and C_R tells us the misalignment *in camera space*.
        # But we need to adjust the *source overlap*.
        
        # Alternative Strategy (Visual Alignment):
        # We want the features to align perfectly on the screen.
        # If we project them with the WRONG overlap, the features will be double (ghosting) or missing.
        # Wait, the "Auto-Calibration" usually means finding the geometric relationship.
        
        # Let's try a simpler approach closer to the user's prompt "detects the overlap".
        # Maybe we project a calibration pattern?
        # Or we just use the current content.
        
        # If we project the LEFT image and RIGHT image separately.
        # We find the bounding box of the projected area in the camera frame.
        # We can estimate the pixel-per-mm or pixel-per-projector-pixel ratio if we know the total resolution.
        
        # Let's stick to a robust method:
        # 1. Detect features in C_L (Left Proj) and C_R (Right Proj).
        # 2. Match them.
        # 3. The translation vector (tx) between them in camera space represents the physical distance between the two projectors' projections of the *same* content features?
        # No, if we project the *split* images, the content is already different except for the overlap.
        # So features in the overlap region of the Source Image should appear at the SAME location on the screen (and thus in the camera) if calibrated correctly.
        # If they are NOT calibrated, they will appear at different locations.
        # So:
        # 1. Project Left Image (with current overlap guess).
        # 2. Project Right Image (with current overlap guess).
        # 3. Capture both.
        # 4. Find matches. Ideally, for the *same* feature in the source image, the point in C_L and C_R should be identical (distance = 0).
        # 5. If there is a shift (dx), that's the error.
        # 6. We convert dx (camera pixels) to projector pixels and adjust overlap.
        
        # To do this, we need to know the scale (Projector Pixels / Camera Pixels).
        # We can estimate this by detecting the screen corners or using a known pattern size.
        # For now, let's use a simple proportional controller or just report the pixel error.
        
        # Let's calculate the average horizontal shift of the matches.
        matches_mask = mask.ravel().tolist()
        dx_list = []
        for i, match in enumerate(good_matches):
            if matches_mask[i]:
                # pt_left is the position of the feature in the Left-Projector-Only capture
                # pt_right is the position of the feature in the Right-Projector-Only capture
                p1 = kp1[match.queryIdx].pt
                p2 = kp2[match.trainIdx].pt
                # We want p1 and p2 to be the same if aligned.
                dx = p2[0] - p1[0]
                dx_list.append(dx)
        
        if not dx_list:
            return 0
            
        avg_dx = np.median(dx_list)
        return avg_dx

    def simulate_calibration(self, image_width: int, current_overlap: int) -> int:
        """
        Simulates the calibration process.
        Returns the suggested adjustment to the overlap.
        """
        # Simulation logic:
        # Assume the "perfect" overlap is 100 pixels.
        # If current is 75, we are off by 25.
        # We return the error.
        target = 100
        error = target - current_overlap
        # Add some noise
        noise = np.random.randint(-2, 3)
        return error + noise
