import configparser
from pathlib import Path
from typing import Any, Union

class ConfigReader:
    """
    Handles the parsing and retrieval of application settings from INI files.
    """

    def __init__(self, config_path: str):
        self.file_path = Path(config_path)
        if not self.file_path.exists():
            raise FileNotFoundError(f"Configuration file not found: {self.file_path}")

        self._parser = configparser.ConfigParser()
        self._parser.read(self.file_path)

    def _get(self, section: str, key: str, fallback: Any = None) -> str:
        try:
            return self._parser.get(section, key)
        except (configparser.NoSectionError, configparser.NoOptionError):
            return fallback

    def get_value(self, section: str, key: str, fallback: Any = None) -> Union[str, Any]:
        return self._get(section, key, fallback)

    def get_linear_parameter(self) -> float:
        val = self._get('Parameters', 'linear_parameter', '0.0')
        return float(val)

    def get_image_path(self) -> str:
        return self._get('Parameters', 'image_path', '')

    def get_overlap(self) -> str:
        return self._get('Parameters', 'overlap', '0')

    def get_blend_mode(self) -> str:
        return self._get('Parameters', 'blend_mode', 'linear')

    def save_config(self) -> None:
        with open(self.file_path, 'w') as configfile:
            self._parser.write(configfile)