import os

# ---- DISABLE PYTORCH MPS (Apple GPU) ----
os.environ["PYTORCH_ENABLE_MPS_FALLBACK"] = "1"
os.environ["PYTORCH_MPS_DISABLE"] = "1"
# ----------------------------------------

import tkinter as tk
from tkinter import ttk, messagebox
from PIL import Image, ImageTk
import cv2
import numpy as np

# Local imports (must come AFTER disabling MPS)
import altmain
import config_reader


class ImageDisplayApp:
    """
    Main GUI application for controlling projection blending (Static Image Only).
    """

    def __init__(self, root: tk.Tk):
        self.root = root
        self.setup_window()

        # Initialize Logic
        try:
            self.cfg = config_reader.ConfigReader("config.ini")
            self.processor = altmain.ProjectionSplit()
        except Exception as e:
            messagebox.showerror("Initialization Error", str(e))
            return

        self._update_job = None
        self.labels = {}
        self.image_paths = {
            "main": self.cfg.get_image_path(),
            "left": "left.png",
            "right": "right.png"
        }

        self.setup_ui()

    def setup_window(self):
        self.root.title("Projection Blending Controller")
        self.root.geometry("1000x600")
        self.root.configure(bg="#f5f5f5")

    def setup_ui(self):
        # Image Display Area
        self.image_frame = ttk.Frame(self.root, padding=10)
        self.image_frame.pack(pady=20)

        for key in ["main", "left", "right"]:
            lbl = ttk.Label(self.image_frame)
            lbl.pack(side=tk.LEFT, padx=10)
            self.labels[key] = lbl

        # Controls
        ctrl_frame = ttk.Frame(self.root, padding=10)
        ctrl_frame.pack(pady=10)

        ttk.Label(ctrl_frame, text="Overlap (px):").grid(row=0, column=0, padx=5)
        self.param_var = tk.IntVar(value=int(self.cfg.get_overlap()))
        overlap_entry = ttk.Entry(ctrl_frame, textvariable=self.param_var, width=8)
        overlap_entry.grid(row=0, column=1, padx=5)

        ttk.Label(ctrl_frame, text="Blend Mode:").grid(row=0, column=2, padx=5)
        self.blend_var = tk.StringVar(value=self.cfg.get_blend_mode())
        blend_box = ttk.Combobox(
            ctrl_frame,
            textvariable=self.blend_var,
            values=["linear", "quadratic", "gaussian"],
            state="readonly",
            width=12
        )
        blend_box.grid(row=0, column=3, padx=5)

        overlap_entry.bind("<FocusOut>", lambda e: self.debounced_update())
        blend_box.bind("<<ComboboxSelected>>", lambda e: self.debounced_update())

        btn_frame = ttk.Frame(self.root, padding=10)
        btn_frame.pack(pady=5)

        ttk.Button(btn_frame, text="Left Fullscreen",
                   command=lambda: self.show_fullscreen("left")).pack(side=tk.LEFT, padx=10)
        ttk.Button(btn_frame, text="Right Fullscreen",
                   command=lambda: self.show_fullscreen("right")).pack(side=tk.LEFT, padx=10)
        ttk.Button(btn_frame, text="Auto-Calibrate",
                   command=self.run_calibration).pack(side=tk.LEFT, padx=10)

    def debounced_update(self):
        if self._update_job:
            self.root.after_cancel(self._update_job)
        self._update_job = self.root.after(500, self.run_processing_update)

    def run_processing_update(self):
        try:
            overlap = self.param_var.get()
            blend = self.blend_var.get()

            self.processor.process_images(overlap, blend)

            self.update_display({
                "left": self.processor.image_left,
                "right": self.processor.image_right,
                "main": self.processor.image_main
            })
        except Exception as e:
            print(f"Processing Error: {e}")

    def update_display(self, images_dict):
        for key, np_img in images_dict.items():
            if np_img is None:
                continue

            if np_img.shape[2] == 4:
                cvt_img = cv2.cvtColor(np_img, cv2.COLOR_BGRA2RGBA)
            else:
                cvt_img = cv2.cvtColor(np_img, cv2.COLOR_BGR2RGB)

            pil_img = Image.fromarray(cvt_img).resize((300, 300))
            photo = ImageTk.PhotoImage(pil_img)

            self.labels[key].configure(image=photo, text="")
            self.labels[key].image = photo

    def show_fullscreen(self, key):
        path = self.image_paths.get(key)
        if not os.path.exists(path):
            messagebox.showwarning("File Missing", f"{path} not found.")
            return

        top = tk.Toplevel(self.root)
        top.attributes("-fullscreen", True)
        top.configure(bg="black")
        top.bind("<Escape>", lambda e: top.destroy())

        lbl = ttk.Label(top, background="black")
        lbl.pack(expand=True)

        def _refresh():
            if not top.winfo_exists():
                return
            try:
                pil_img = Image.open(path)
                photo = ImageTk.PhotoImage(pil_img)
                lbl.configure(image=photo)
                lbl.image = photo
            except:
                pass
            top.after(100, _refresh)

        _refresh()

    def run_calibration(self):
        try:
            import calibration
            calib = calibration.CalibrationManager()

            current_overlap = self.param_var.get()
            adjustment = calib.simulate_calibration(1920, current_overlap)

            new_overlap = current_overlap + adjustment
            self.param_var.set(new_overlap)
            self.debounced_update()

            messagebox.showinfo(
                "Calibration Complete",
                f"Overlap adjusted by {adjustment}px.\nNew Overlap: {new_overlap}"
            )
        except Exception as e:
            messagebox.showerror("Calibration Error", str(e))


if __name__ == "__main__":
    root = tk.Tk()
    app = ImageDisplayApp(root)
    root.mainloop()
