## @file altmain.py
## @brief Core image splitting and blending module for dual-projector setups.

import cv2
import numpy as np
from scipy.special import erf
import config_reader
from typing import Tuple, Optional

## @class ProjectionSplit
## @brief Handles splitting images for dual-projector setups with overlap blending.
##
## @details This class is responsible for:
## - Loading and processing input images for dual-projector setups
## - Generating mathematical alpha blending curves for smooth transitions
## - Splitting images into left/right sections with configurable overlap
## - Applying blending using pure NumPy operations (no GPU dependency)
## - Saving processed outputs to disk for projector display
class ProjectionSplit:
    
    ## @brief Initializes the ProjectionSplit processor with NumPy backend.
    ##
    ## @details Constructor that sets up the processor by:
    ## - Initializing image storage buffers as None
    ## - Loading configuration from config.ini file
    ## - Using pure NumPy operations for cross-platform compatibility
    ##
    ## @post self.image_left, self.image_right, self.image_main are initialized to None
    ## @post self.cfg contains loaded configuration from config.ini
    ##
    ## @throws FileNotFoundError if config.ini cannot be found
    def __init__(self):
        ## @var image_left
        ## @brief Processed left projector output image
        self.image_left: Optional[np.ndarray] = None
        
        ## @var image_right
        ## @brief Processed right projector output image
        self.image_right: Optional[np.ndarray] = None
        
        ## @var image_main
        ## @brief Original input image buffer for reference
        self.image_main: Optional[np.ndarray] = None
        
        ## @var cfg
        ## @brief Configuration manager instance
        self.cfg = config_reader.ConfigReader("config.ini")
        
        print("ProjectionSplit initialized (NumPy backend)")


    ## @brief Generates alpha blending curves for overlap region.
    ##
    ## @details Creates mathematical curves that determine transparency values
    ## across the overlap zone using three supported blending types:
    ## - Linear: Straight-line fade with configurable slope parameter
    ## - Quadratic: Parabolic fade (x²) for smoother transition
    ## - Gaussian: Bell curve fade using error function for natural blending
    ##
    ## @param overlap Width of the overlap region in pixels
    ## @param blend_type Type of blending curve: "linear", "quadratic", or "gaussian"
    ##
    ## @return Tuple of (left_curve, right_curve) where each is a 1D numpy array
    ##         of alpha values (float32) ranging from 0.0 to 1.0 across overlap width
    ##
    ## @throws ValueError if an unsupported blend_type is provided
    def _generate_alpha_curves(
        self, overlap: int, blend_type: str
    ) -> Tuple[np.ndarray, np.ndarray]:
        x = np.linspace(0, 1, overlap)

        if blend_type == "linear":
            param = self.cfg.get_linear_parameter()
            left = 1 - param * x
            right = 1 - param + param * x

        elif blend_type == "quadratic":
            left = (1 - x) ** 2
            right = x ** 2

        elif blend_type == "gaussian":
            sigma = 0.25
            g = 0.5 * (1 + erf((x - 0.5) / (sigma * np.sqrt(2))))
            left = 1 - g
            right = g
        else:
            raise ValueError(f"Unsupported blend_type: {blend_type}")

        return left.astype(np.float32), right.astype(np.float32)


    ## @brief Applies alpha blending to split images using NumPy operations.
    ##
    ## @details Core processing method that:
    ## 1. Converts 3-channel images to 4-channel (BGRA) if needed
    ## 2. Splits image into left and right sections with specified overlap
    ## 3. Generates alpha blending curves
    ## 4. Applies curves to alpha channels of overlap regions
    ## 5. Stores results in instance buffers and saves to disk
    ##
    ## @param image Input image as numpy array (BGR or BGRA format)
    ## @param overlap Width of overlap region in pixels (must be even for symmetric split)
    ## @param blend_type Type of blending curve to apply
    ##
    ## @post self.image_left and self.image_right contain processed images
    ## @post left.png and right.png are saved to working directory
    def _apply_blending(self, image: np.ndarray, overlap: int, blend_type: str) -> None:
        if image.shape[2] == 3:
            image = cv2.cvtColor(image, cv2.COLOR_BGR2BGRA)

        height, width = image.shape[:2]
        split_x = width // 2

        left_end = split_x + overlap // 2
        right_start = split_x - overlap // 2

        left_img = image[:, :left_end].astype(np.float32)
        right_img = image[:, right_start:].astype(np.float32)

        alpha_l, alpha_r = self._generate_alpha_curves(overlap, blend_type)

        left_img[:, -overlap:, 3] *= alpha_l[None, :]
        right_img[:, :overlap, 3] *= alpha_r[None, :]

        self.image_left = left_img.astype(np.uint8)
        self.image_right = right_img.astype(np.uint8)

        cv2.imwrite("left.png", self.image_left)
        cv2.imwrite("right.png", self.image_right)


    ## @brief Main public interface for processing static images.
    ##
    ## @details Loads input image from disk and applies splitting and blending operations.
    ## This is the primary method called by the GUI controller for image processing.
    ##
    ## @param overlap Overlap width in pixels (default: 75)
    ## @param blend_type Blending curve type (default: "linear")
    ##
    ## @throws FileNotFoundError if input.png cannot be found in working directory
    ## @throws ValueError if blend_type is not supported
    ##
    ## @post self.image_main contains the loaded input image
    ## @post self.image_left and self.image_right contain processed outputs
    ## @post left.png and right.png are saved to working directory
    def process_images(self, overlap: int = 75, blend_type: str = "linear"):
        image = cv2.imread("input.png", cv2.IMREAD_UNCHANGED)
        if image is None:
            raise FileNotFoundError("input.png not found.")
        self.image_main = image
        self._apply_blending(image, overlap, blend_type)