## @file config_reader.py
#  @brief Configuration file reader using INI format
#
#  This module provides a class for reading and managing configuration files
#  in INI format, specifically designed for handling image processing parameters.

import configparser
from pathlib import Path
from typing import Any, Union

## @class ConfigReader
#  @brief A configuration file reader for INI-format files
#
#  @details This class provides a convenient interface for reading configuration
#  parameters from INI files. It uses Python's configparser module internally
#  and provides type-safe accessor methods for different parameter types.
#  The class is particularly designed for managing image processing configurations
#  including linear parameters, image paths, overlap values, and blend modes.
#
#  The configuration file is expected to have a 'Parameters' section with
#  keys such as 'linear_parameter', 'image_path', 'overlap', and 'blend_mode'.
#
#  @note All getter methods provide default fallback values if the configuration
#  keys are missing.
class ConfigReader:

    ## @brief Constructor for ConfigReader
    #  @param config_path Path to the configuration file (string)
    #  @throws FileNotFoundError if the configuration file doesn't exist
    #
    #  @details Initializes the ConfigReader by loading the specified INI file.
    #  The file path is converted to a Path object for robust file handling.
    #  The configuration is immediately read into memory using configparser.
    #
    #  Example usage:
    #  @code
    #  reader = ConfigReader('/path/to/config.ini')
    #  @endcode
    def __init__(self, config_path: str):
        ## @brief Path object representing the configuration file location
        self.file_path = Path(config_path)  
        
        if not self.file_path.exists():
            raise FileNotFoundError(f"Configuration file not found: {self.file_path}")

        ## @brief Internal ConfigParser instance for reading INI files
        #  @private
        self._parser = configparser.ConfigParser()  
        self._parser.read(self.file_path)


    ## @brief Retrieves the linear parameter value
    #  @return Linear parameter as a float
    #
    #  @details Reads the 'linear_parameter' key from the 'Parameters' section.
    #  If the key doesn't exist, returns 0.0 as the default value.
    #  The returned value is guaranteed to be a float type.
    #
    #  Example:
    #  @code
    #  linear_val = reader.get_linear_parameter()
    #  print(f"Linear parameter: {linear_val}")
    #  @endcode
    def get_linear_parameter(self) -> float:
        val = self._get('Parameters', 'linear_parameter', '0.0')
        return float(val)


    ## @brief Retrieves the image path
    #  @return Image path as a string
    #
    #  @details Reads the 'image_path' key from the 'Parameters' section.
    #  Returns an empty string if the key doesn't exist.
    def get_image_path(self) -> str:
        return self._get('Parameters', 'image_path', '')


    ## @brief Retrieves the overlap value
    #  @return Overlap value as a string
    #
    #  @details Reads the 'overlap' key from the 'Parameters' section.
    #  Returns '0' as the default value if the key doesn't exist.
    def get_overlap(self) -> str:
        return self._get('Parameters', 'overlap', '0')


    ## @brief Retrieves the blend mode
    #  @return Blend mode as a string
    #
    #  @details Reads the 'blend_mode' key from the 'Parameters' section.
    #  Returns 'linear' as the default value if the key doesn't exist.
    #  Common blend modes might include 'linear', 'average', 'max', etc.
    def get_blend_mode(self) -> str:
        return self._get('Parameters', 'blend_mode', 'linear')

    ## @brief Saves the current configuration back to the file
    #  @return None
    #
    #  @details Writes the current state of the configuration parser back to
    #  the original file. This is useful if the configuration has been modified
    #  programmatically and needs to be persisted.
    #
    #  @warning This will overwrite the existing configuration file.
    def save_config(self) -> None:
        with open(self.file_path, 'w') as configfile:
            self._parser.write(configfile)

    ## @brief Internal helper method to safely retrieve configuration values
    #  @param section The section name in the INI file
    #  @param key The key name within the section
    #  @param fallback Default value to return if section or key doesn't exist
    #  @return Configuration value as a string, or fallback value
    #  @private
    #
    #  @details This method abstracts the error handling for missing sections
    #  or keys in the configuration file. It catches NoSectionError and
    #  NoOptionError exceptions and returns the fallback value instead.
    def _get(self, section: str, key: str, fallback: Any = None) -> str:
        try:
            return self._parser.get(section, key)
        except (configparser.NoSectionError, configparser.NoOptionError):
            return fallback