## @file ui.py
#  @brief GUI application for projection blending control and visualization.
#  This module provides a Tkinter-based graphical interface for controlling
#  projection blending parameters and displaying processed images.

import os
import tkinter as tk
from tkinter import ttk, messagebox
from PIL import Image, ImageTk
import cv2
import numpy as np
import altmain
import config_reader


## @class ImageDisplayApp
#  @brief Main application class for projection blending controller.
#
#  @details
#  The ImageDisplayApp class creates a GUI application that allows users to:
#  - View main, left, and right projection images
#  - Adjust overlap and blend mode parameters
#  - Display images in fullscreen mode
#  - Run auto-calibration for optimal projection settings
#
#  The application uses configuration files to load initial settings and
#  processes images in real-time based on user input.
#
#  @section usage Usage Example
#  @code{.py}
#  root = tk.Tk()
#  app = ImageDisplayApp(root)
#  root.mainloop()
#  @endcode
class ImageDisplayApp:

    ## @brief Constructor that initializes the application.
    #
    #  @details
    #  Initializes the projection blending controller application by:
    #  - Setting up the main window
    #  - Loading configuration from config.ini
    #  - Creating the image processor
    #  - Building the user interface
    #
    #  @param root The Tkinter root window instance
    #
    #  @throws Exception If configuration file is missing or initialization fails
    #
    #  @public
    def __init__(self, root: tk.Tk):
        ## @brief The main Tkinter root window
        #  @public
        self.root = root          
        self.setup_window()
        
        try:
            ## @brief Configuration reader instance for loading settings
            #  @public
            self.cfg = config_reader.ConfigReader("config.ini")  
            
            ## @brief Image processor instance for handling projections
            #  @public
            self.processor = altmain.ProjectionSplit()          
        except Exception as e:
            messagebox.showerror("Initialization Error", str(e))
            return

        ## @brief Job ID for debounced update scheduling
        #  @details Used to cancel pending updates when new changes occur
        #  @protected
        self._update_job = None  
        
        ## @brief Dictionary of image label widgets
        #  @details Maps keys ("main", "left", "right") to their Label widgets
        #  @public
        self.labels = {}          
        
        ## @brief Dictionary of image file paths
        #  @details Maps image types to their file paths
        #  @public
        self.image_paths = {
            "main": self.cfg.get_image_path(),  
            "left": "left.png",                  
            "right": "right.png"              
        }

        self.setup_ui()


    ## @brief Sets up the main application window properties.
    #
    #  @details
    #  Configures the window title, size, and background color.
    #
    #  @return None
    #
    #  @public
    def setup_window(self):
        self.root.title("Projection Blending Controller")
        self.root.geometry("1000x600")
        self.root.configure(bg="#f5f5f5")


    ## @brief Creates and arranges all UI components.
    #
    #  @details
    #  Sets up the user interface including:
    #  - Image display labels for main, left, and right projections
    #  - Control widgets for overlap and blend mode
    #  - Action buttons for fullscreen and calibration
    #  - Event bindings for parameter updates
    #
    #  @return None
    #
    #  @public
    def setup_ui(self):
        self.image_frame = ttk.Frame(self.root, padding=10)
        self.image_frame.pack(pady=20)

        for key in ["main", "left", "right"]:
            lbl = ttk.Label(self.image_frame)
            lbl.pack(side=tk.LEFT, padx=10)
            self.labels[key] = lbl

        ctrl_frame = ttk.Frame(self.root, padding=10)
        ctrl_frame.pack(pady=10)

        ttk.Label(ctrl_frame, text="Overlap (px):").grid(row=0, column=0, padx=5)
        
        ## @brief Integer variable for overlap parameter
        #  @public
        self.param_var = tk.IntVar(value=int(self.cfg.get_overlap()))
        overlap_entry = ttk.Entry(ctrl_frame, textvariable=self.param_var, width=8)
        overlap_entry.grid(row=0, column=1, padx=5)

        ttk.Label(ctrl_frame, text="Blend Mode:").grid(row=0, column=2, padx=5)
        
        ## @brief String variable for blend mode selection
        #  @public
        self.blend_var = tk.StringVar(value=self.cfg.get_blend_mode())
        blend_box = ttk.Combobox(
            ctrl_frame,
            textvariable=self.blend_var,
            values=["linear", "quadratic", "gaussian"],
            state="readonly",
            width=12
        )
        blend_box.grid(row=0, column=3, padx=5)

        overlap_entry.bind("<FocusOut>", lambda e: self.debounced_update())
        blend_box.bind("<<ComboboxSelected>>", lambda e: self.debounced_update())

        btn_frame = ttk.Frame(self.root, padding=10)
        btn_frame.pack(pady=5)

        ttk.Button(btn_frame, text="Left Fullscreen",
                   command=lambda: self.show_fullscreen("left")).pack(side=tk.LEFT, padx=10)
        ttk.Button(btn_frame, text="Right Fullscreen",
                   command=lambda: self.show_fullscreen("right")).pack(side=tk.LEFT, padx=10)
        ttk.Button(btn_frame, text="Auto-Calibrate",
                   command=self.run_calibration).pack(side=tk.LEFT, padx=10)


    ## @brief Schedules a debounced update after parameter changes.
    #
    #  @details
    #  Cancels any pending updates and schedules a new update after 500ms delay.
    #  This prevents excessive processing during rapid parameter changes.
    #
    #  @return None
    #
    #  @protected
    def debounced_update(self):
        if self._update_job:
            self.root.after_cancel(self._update_job)
        self._update_job = self.root.after(500, self.run_processing_update)


    ## @brief Executes image processing with current parameters.
    #
    #  @details
    #  Retrieves current overlap and blend mode values, processes the images,
    #  and updates the display with the results.
    #
    #  @return None
    #
    #  @throws Exception Prints error message if processing fails
    #
    #  @public
    def run_processing_update(self):
        try:
            overlap = self.param_var.get()
            blend = self.blend_var.get()

            self.processor.process_images(overlap, blend)

            self.update_display({
                "left": self.processor.image_left,
                "right": self.processor.image_right,
                "main": self.processor.image_main
            })
        except Exception as e:
            print(f"Processing Error: {e}")


    ## @brief Updates the GUI with processed images.
    #
    #  @details
    #  Converts numpy arrays to PhotoImage objects and displays them
    #  in the corresponding label widgets. Handles both BGR and BGRA formats.
    #
    #  @param images_dict Dictionary mapping image keys to numpy arrays
    #
    #  @return None
    #
    #  @public
    def update_display(self, images_dict):
        for key, np_img in images_dict.items():
            if np_img is None:
                continue

            if np_img.shape[2] == 4:
                cvt_img = cv2.cvtColor(np_img, cv2.COLOR_BGRA2RGBA)
            else:
                cvt_img = cv2.cvtColor(np_img, cv2.COLOR_BGR2RGB)

            pil_img = Image.fromarray(cvt_img).resize((300, 300))
            photo = ImageTk.PhotoImage(pil_img)

            self.labels[key].configure(image=photo, text="")
            self.labels[key].image = photo

    ## @brief Displays an image in fullscreen mode.
    #
    #  @details
    #  Opens a fullscreen window displaying the specified image.
    #  The window can be closed by pressing the Escape key.
    #  The image refreshes every 100ms to show updates.
    #
    #  @param key The image key ("main", "left", or "right")
    #
    #  @return None
    #
    #  @warning Shows warning message if image file doesn't exist
    #
    #  @public
    def show_fullscreen(self, key):
        path = self.image_paths.get(key)
        if not os.path.exists(path):
            messagebox.showwarning("File Missing", f"{path} not found.")
            return

        top = tk.Toplevel(self.root)
        top.attributes("-fullscreen", True)
        top.configure(bg="black")
        top.bind("<Escape>", lambda e: top.destroy())

        lbl = ttk.Label(top, background="black")
        lbl.pack(expand=True)

        def _refresh():
            if not top.winfo_exists():
                return
            try:
                pil_img = Image.open(path)
                photo = ImageTk.PhotoImage(pil_img)
                lbl.configure(image=photo)
                lbl.image = photo
            except:
                pass
            top.after(100, _refresh)

        _refresh()


    ## @brief Runs the auto-calibration process.
    #
    #  @details
    #  Executes calibration to automatically adjust the overlap parameter
    #  for optimal projection alignment. Updates the overlap value and
    #  triggers image reprocessing.
    #
    #  @return None
    #
    #  @throws Exception Shows error dialog if calibration fails
    #
    #  @public
    def run_calibration(self):
        try:
            import calibration
            calib = calibration.CalibrationManager()

            current_overlap = self.param_var.get()
            adjustment = calib.simulate_calibration(1920, current_overlap)

            new_overlap = current_overlap + adjustment
            self.param_var.set(new_overlap)
            self.debounced_update()

            messagebox.showinfo(
                "Calibration Complete",
                f"Overlap adjusted by {adjustment}px.\nNew Overlap: {new_overlap}"
            )
        except Exception as e:
            messagebox.showerror("Calibration Error", str(e))


## @brief Main entry point for the application.
#
#  @details
#  Creates the Tkinter root window, initializes the ImageDisplayApp,
#  and starts the main event loop.
if __name__ == "__main__":
    root = tk.Tk()
    app = ImageDisplayApp(root)
    root.mainloop()